<?php
header('Content-Type: application/json; charset=utf-8');

// Ativar exibição de erros para depuração (remover em produção)
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

// Iniciar sessão
session_start();

// Função para enviar resposta de erro
function sendError($status, $error, $details = '') {
    http_response_code($status);
    $response = ['success' => false, 'error' => $error];
    if ($details) {
        $response['details'] = $details;
    }
    error_log("Erro em clients.php: $error - $details");
    echo json_encode($response);
    exit;
}

// Verificar inclusão de config.php
if (!file_exists('config.php')) {
    sendError(500, 'Arquivo de configuração não encontrado', 'config.php ausente');
}
require_once 'config.php';

// Função para validar CPF
function validateCPF($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    if (strlen($cpf) != 11) return false;
    if (preg_match('/(\d)\1{10}/', $cpf)) return false;
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) return false;
    }
    return true;
}

try {
    // Verificar conexão com o banco
    $conn = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    error_log('Conexão com o banco de dados estabelecida com sucesso');

    $method = $_SERVER['REQUEST_METHOD'];
    $input = json_decode(file_get_contents('php://input'), true);

    if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'check_auth') {
        // VERIFICAR AUTENTICAÇÃO
        if (!isset($_SESSION['user_id'])) {
            sendError(401, 'Não autenticado', 'Sessão não encontrada');
        }
        echo json_encode(['success' => true, 'user_id' => $_SESSION['user_id']]);
        exit;
    } elseif ($method === 'GET') {
        // LISTAR CLIENTES - EXIGE LOGIN
        if (!isset($_SESSION['user_id'])) {
            sendError(401, 'Não autenticado', 'Sessão não encontrada');
        }
        $stmt = $conn->prepare("
            SELECT c.*, p.name AS plan_name,
                   si.name AS equipment_name
            FROM clients c
            LEFT JOIN plans p ON c.planId = p.id
            LEFT JOIN stock_movements sm ON c.cpf = sm.client_id AND sm.type = 'exit'
            LEFT JOIN stock_items si ON sm.item_id = si.id
            WHERE sm.id = (
                SELECT MAX(sm2.id)
                FROM stock_movements sm2
                WHERE sm2.client_id = c.cpf AND sm2.type = 'exit'
            ) OR sm.id IS NULL
            ORDER BY c.name
        ");
        $stmt->execute();
        $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode(['success' => true, 'data' => $clients]);
    } elseif ($method === 'POST') {
        // CADASTRAR CLIENTE - SEM LOGIN
        if (!isset($input['cpf']) || !isset($input['serial']) || !isset($input['name']) || !isset($input['address']) || !isset($input['planId']) || !isset($input['installer']) || !isset($input['pppoe']) || !isset($input['password'])) {
            sendError(400, 'Todos os campos obrigatórios (CPF, Serial, Nome, Endereço, Plano, Instalador, PPPoE, Senha) devem ser preenchidos');
        }
        if (!validateCPF($input['cpf'])) {
            sendError(400, 'CPF inválido');
        }
        if ($input['phone'] && strlen(preg_replace('/[^0-9]/', '', $input['phone'])) < 10) {
            sendError(400, 'Telefone deve conter pelo menos 10 dígitos');
        }
        if ($input['dueDay'] && !in_array($input['dueDay'], [10, 20, 30])) {
            sendError(400, 'Dia de vencimento deve ser 10, 20 ou 30');
        }
        if ($input['city'] && !in_array($input['city'], ['Ipixuna', 'Eirunepé', 'Carauari', 'Itamarati'])) {
            sendError(400, 'Cidade inválida');
        }
        $stmt = $conn->prepare("
            INSERT INTO clients (cpf, serial, name, address, number, complement, city, dueDay, phone, birthDate, observation, planId, installer, pppoe, password, active)
            VALUES (:cpf, :serial, :name, :address, :number, :complement, :city, :dueDay, :phone, :birthDate, :observation, :planId, :installer, :pppoe, :password, :active)
        ");
        $stmt->execute([
            'cpf' => preg_replace('/[^0-9]/', '', $input['cpf']),
            'serial' => $input['serial'],
            'name' => $input['name'],
            'address' => $input['address'],
            'number' => $input['number'] ?? null,
            'complement' => $input['complement'] ?? null,
            'city' => $input['city'] ?? null,
            'dueDay' => $input['dueDay'] ?? null,
            'phone' => $input['phone'] ? preg_replace('/[^0-9]/', '', $input['phone']) : null,
            'birthDate' => $input['birthDate'] ?: null,
            'observation' => $input['observation'] ?? null,
            'planId' => $input['planId'],
            'installer' => $input['installer'],
            'pppoe' => $input['pppoe'],
            'password' => $input['password'],
            'active' => isset($input['active']) ? (int)$input['active'] : 1
        ]);
        echo json_encode(['success' => true, 'message' => 'Cliente cadastrado com sucesso']);
    } elseif ($method === 'PUT') {
        // EDITAR CLIENTE - EXIGE LOGIN
        if (!isset($_SESSION['user_id'])) {
            sendError(401, 'Não autenticado', 'Sessão não encontrada');
        }
        if (!isset($_GET['cpf']) || !isset($input['serial']) || !isset($input['name']) || !isset($input['address']) || !isset($input['planId']) || !isset($input['installer']) || !isset($input['pppoe']) || !isset($input['password'])) {
            sendError(400, 'Todos os campos obrigatórios (CPF, Serial, Nome, Endereço, Plano, Instalador, PPPoE, Senha) devem ser preenchidos');
        }
        if ($input['phone'] && strlen(preg_replace('/[^0-9]/', '', $input['phone'])) < 10) {
            sendError(400, 'Telefone deve conter pelo menos 10 dígitos');
        }
        if ($input['dueDay'] && !in_array($input['dueDay'], [10, 20, 30])) {
            sendError(400, 'Dia de vencimento deve ser 10, 20 ou 30');
        }
        if ($input['city'] && !in_array($input['city'], ['Ipixuna', 'Eirunepé', 'Carauari', 'Itamarati'])) {
            sendError(400, 'Cidade inválida');
        }
        $stmt = $conn->prepare("
            UPDATE clients
            SET serial = :serial, name = :name, address = :address, number = :number, complement = :complement,
                city = :city, dueDay = :dueDay, phone = :phone, birthDate = :birthDate, observation = :observation,
                planId = :planId, installer = :installer, pppoe = :pppoe, password = :password, active = :active
            WHERE cpf = :cpf
        ");
        $stmt->execute([
            'cpf' => preg_replace('/[^0-9]/', '', $_GET['cpf']),
            'serial' => $input['serial'],
            'name' => $input['name'],
            'address' => $input['address'],
            'number' => $input['number'] ?? null,
            'complement' => $input['complement'] ?? null,
            'city' => $input['city'] ?? null,
            'dueDay' => $input['dueDay'] ?? null,
            'phone' => $input['phone'] ? preg_replace('/[^0-9]/', '', $input['phone']) : null,
            'birthDate' => $input['birthDate'] ?: null,
            'observation' => $input['observation'] ?? null,
            'planId' => $input['planId'],
            'installer' => $input['installer'],
            'pppoe' => $input['pppoe'],
            'password' => $input['password'],
            'active' => isset($input['active']) ? (int)$input['active'] : 1
        ]);
        echo json_encode(['success' => true, 'message' => 'Cliente atualizado com sucesso']);
    } elseif ($method === 'PATCH' && isset($_GET['action']) && $_GET['action'] === 'update_active') {
        // ATUALIZAR STATUS ATIVO - EXIGE LOGIN
        if (!isset($_SESSION['user_id'])) {
            sendError(401, 'Não autenticado', 'Sessão não encontrada');
        }
        if (!isset($input['cpf']) || !isset($input['active'])) {
            sendError(400, 'CPF e active são obrigatórios');
        }
        $stmt = $conn->prepare("UPDATE clients SET active = ? WHERE cpf = ?");
        $stmt->execute([(int)$input['active'], preg_replace('/[^0-9]/', '', $input['cpf'])]);
        echo json_encode(['success' => true, 'message' => 'Status atualizado com sucesso']);
    } elseif ($method === 'DELETE') {
        // EXCLUIR CLIENTE - EXIGE LOGIN
        if (!isset($_SESSION['user_id'])) {
            sendError(401, 'Não autenticado', 'Sessão não encontrada');
        }
        if (!isset($_GET['cpf'])) {
            sendError(400, 'CPF é obrigatório');
        }
        $stmt = $conn->prepare("DELETE FROM clients WHERE cpf = ?");
        $stmt->execute([preg_replace('/[^0-9]/', '', $_GET['cpf'])]);
        echo json_encode(['success' => true, 'message' => 'Cliente excluído com sucesso']);
    } else {
        sendError(405, 'Método não permitido');
    }
} catch (PDOException $e) {
    sendError(500, 'Erro no banco de dados', $e->getMessage());
} catch (Exception $e) {
    sendError(500, 'Erro geral', $e->getMessage());
}
?>